<?php
/*
Plugin Name: Sheet2DB
Plugin URI: https://sheet2db.com/docs/wordpress-plugin
Description: The Sheet2DB WordPress plugin enables you to seamlessly integrate and display content from Google Spreadsheets on your WordPress site.
Version: 1.0.1
Author: Sheet2DB
Author URI: https://sheet2db.com
*/

function sheet2db_enqueue_styles() {
    // Check if the flag for loading table styles is set
    if (!is_admin() && did_action('wp_footer')) {
        // Enqueue the Tailwind CSS from the provided URL
        wp_enqueue_style('sheet2db-tailwind', 'https://sheet2db.com/table.tailwind.css', [], null);
        // Enqueue FontAwesome for icons
        wp_enqueue_style('sheet2db-fontawesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css', [], null);
    }
}
add_action('wp_enqueue_scripts', 'sheet2db_enqueue_styles');

function sheet2db_shortcode($atts, $content = null) {
    $atts = shortcode_atts([
        'url' => '',
    ], $atts);

    if (empty($atts['url'])) {
        return '<p>No URL provided.</p>';
    }

    $cache_key = 'sheet2db_' . md5($atts['url']);
    $data = wp_cache_get($cache_key);

    if ($data === false) {
        $response = wp_remote_get(esc_url_raw($atts['url']));

        if (is_wp_error($response)) {
            return '<p>Failed to retrieve data. Invalid Connection ID</p>';
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return '<p>Invalid JSON data.</p>';
        }

        if (!is_array($data)) {
            return '<p>Data is not in expected format.</p>';
        }

        wp_cache_set($cache_key, $data, '', 3600); // Cache for 1 hour
    }

    $output = '';

    foreach ($data as $item) {
        $output .= preg_replace_callback('/{{([^{}]*)}}/', function ($matches) use ($item) {
            $key = trim($matches[1]);
            return isset($item[$key]) ? esc_html($item[$key]) : esc_html($matches[0]);
        }, $content);
    }

    return $output;
}

function sheet2db_table_shortcode($atts) {
    // Set a flag to indicate that the table shortcode is being used
    add_action('wp_footer', function() {
        wp_enqueue_style('sheet2db-tailwind');
        wp_enqueue_style('sheet2db-fontawesome');
    });

    // Extract shortcode attributes
    $atts = shortcode_atts(array(
        'id' => '',
        'title' => '',
        'table_bg' => '',
        'header_bg' => '',
        'dark_mode' => 'false',
        'font_family' => '',
        'sortable_columns' => '',
        'striped' => 'false',
        'enable_search' => 'true',
        'limit' => '',
        'offset' => '',
        'sheet' => '',
    ), $atts, 'sheet2db_table');

    // Prepare the API URL with parameters
    $url = "https://api.sheet2db.com/v1/" . urlencode($atts['id']);
    $query_params = array();
    foreach (['limit', 'offset', 'sheet'] as $param) {
        if (!empty($atts[$param])) {
            $query_params[$param] = $atts[$param];
        }
    }
    $url .= '?' . http_build_query($query_params);

    // Cache key based on URL
    $cache_key = 'sheet2db_' . md5($url);
    $data = wp_cache_get($cache_key);

    if ($data === false) {
        // Fetch data from the API
        $response = wp_remote_get(esc_url_raw($url));
        if (is_wp_error($response)) {
            return "Error fetching data.";
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (empty($data)) {
            return "No data available.";
        }

        wp_cache_set($cache_key, $data, '', 3600); // Cache for 1 hour
    }

    // Start output buffering
    ob_start();
    ?>

    <?php if ($atts['enable_search'] === 'true'): ?>
        <div class="s2db-p-4">
            <div id="filter-container" class="s2db-relative s2db-mb-4">
                <input type="text" id="filter-input" class="s2db-w-full s2db-p-2 s2db-border s2db-rounded s2db-pl-10" placeholder="Search" oninput="filterTable()">
                <i class="fas fa-search s2db-absolute s2db-left-3 s2db-top-3 s2db-text-gray-500"></i>
            </div>
    <?php endif; ?>
    <div class="s2db-overflow-x-auto">
        <table id="data-table" class="s2db-min-w-full s2db-bg-white s2db-shadow-md s2db-rounded-lg <?php echo esc_attr($atts['table_bg']); ?> <?php echo $atts['dark_mode'] === 'true' ? 's2db-dark:s2db-bg-gray-800' : ''; ?>">
            <thead id="table-header" class="<?php echo esc_attr($atts['header_bg']); ?> <?php echo $atts['dark_mode'] === 'true' ? 's2db-dark:s2db-bg-gray-700' : ''; ?>">
                <tr>
                    <?php
                    // Generate table headers
                    $headers = array_keys($data[0]);
                    foreach ($headers as $header) {
                        $sortable_class = in_array($header, explode(',', $atts['sortable_columns'])) ? 's2db-cursor-pointer s2db-sort-header' : '';
                        echo "<th class='s2db-px-6 s2db-py-3 s2db-text-left s2db-text-xs s2db-font-medium s2db-text-gray-500 s2db-uppercase s2db-tracking-wider s2db-dark:s2db-text-gray-300 " . esc_attr($sortable_class) . "' data-column='" . esc_attr($header) . "' data-sort-direction=''>";
                        echo esc_html($header);
                        if ($sortable_class) {
                            echo " <i class='fas fa-sort'></i>";
                        }
                        echo "</th>";
                    }
                    ?>
                </tr>
            </thead>
            <tbody id="table-body">
                <?php
                // Generate table rows
                foreach ($data as $index => $row) {
                    $row_class = $atts['striped'] === 'true' && $index % 2 === 0 ? 's2db-bg-gray-50 s2db-dark:s2db-bg-gray-700' : '';
                    echo "<tr class='s2db-border-t s2db-dark:s2db-border-gray-600 " . esc_attr($row_class) . "'>";
                    foreach ($row as $cell) {
                        echo "<td class='s2db-px-6 s2db-py-4 s2db-whitespace-nowrap s2db-text-sm s2db-text-gray-700 s2db-dark:s2db-text-gray-300'>" . esc_html($cell) . "</td>";
                    }
                    echo "</tr>";
                }
                ?>
            </tbody>
        </table>
    </div>
    <?php if ($atts['enable_search'] === 'true'): ?>
        </div>
    <?php endif; ?>
    <script>
        document.addEventListener('DOMContentLoaded', function () {
            if ("<?php echo esc_js($atts['dark_mode']); ?>" === 'true') {
                document.body.classList.add('s2db-dark');
            }
            if ("<?php echo esc_js($atts['font_family']); ?>") {
                document.body.style.fontFamily = "<?php echo esc_js($atts['font_family']); ?>";
            }

            // Sort table functionality
            function sortTable(column) {
                const table = document.getElementById('data-table');
                const tbody = table.querySelector('tbody');
                const rows = Array.from(tbody.querySelectorAll('tr'));
                const header = table.querySelector(`th[data-column='${column}']`);
                const isAscending = header.dataset.sortDirection !== 'asc';
                header.dataset.sortDirection = isAscending ? 'asc' : 'desc';
                
                rows.sort((rowA, rowB) => {
                    const cellA = rowA.querySelector(`td:nth-child(${header.cellIndex + 1})`).innerText;
                    const cellB = rowB.querySelector(`td:nth-child(${header.cellIndex + 1})`).innerText;

                    if (!isNaN(Date.parse(cellA)) && !isNaN(Date.parse(cellB))) {
                        return isAscending ? new Date(cellA) - new Date(cellB) : new Date(cellB) - new Date(cellA);
                    } else if (!isNaN(cellA) && !isNaN(cellB)) {
                        return isAscending ? cellA - cellB : cellB - cellA;
                    } else {
                        return isAscending ? cellA.localeCompare(cellB) : cellB.localeCompare(cellA);
                    }
                });

                tbody.innerHTML = '';
                rows.forEach(row => tbody.appendChild(row));

                document.querySelectorAll('.s2db-sort-header').forEach(icon => {
                    if (icon === header) {
                        icon.querySelector('i').className = `fas fa-sort-${isAscending ? 'up' : 'down'}`;
                    } else {
                        icon.querySelector('i').className = 'fas fa-sort';
                    }
                });
            }

            document.querySelectorAll('.s2db-sort-header').forEach(header => {
                header.addEventListener('click', () => sortTable(header.dataset.column));
            });
        });

        function filterTable() {
            const filterInput = document.getElementById('filter-input').value.toLowerCase();
            const rows = document.querySelectorAll('#table-body tr');

            rows.forEach(row => {
                const cells = row.querySelectorAll('td');
                const matches = Array.from(cells).some(cell => cell.textContent.toLowerCase().includes(filterInput));
                row.style.display = matches ? '' : 'none';
            });
        }
    </script>
    <?php
    return ob_get_clean();
}

function sheet2db_cell_shortcode($atts, $content = null) {
    $atts = shortcode_atts([
        'id' => '',
        'sheet' => '',
    ], $atts);

    if (empty($atts['id']) || empty($atts['sheet'])) {
        return '<p>Missing required attributes.</p>';
    }

    // Construct the URL using the provided id and sheet
    $url = "https://api.sheet2db.com/v1/{$atts['id']}?sheet={$atts['sheet']}&format=raw";

    $cache_key = 'sheet2db_' . md5($url);
    $data = wp_cache_get($cache_key);

    if ($data === false) {
        $response = wp_remote_get(esc_url_raw($url));

        if (is_wp_error($response)) {
            return '<p>Failed to retrieve data. Invalid Connection ID.</p>';
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return '<p>Invalid JSON data.</p>';
        }

        if (!is_array($data)) {
            return '<p>Data is not in the expected format.</p>';
        }

        wp_cache_set($cache_key, $data, '', 3600); // Cache for 1 hour
    }

    $sheet_data = $data;
    if (!is_array($sheet_data) || !is_array($sheet_data[0])) {
        return '<p>Sheet data is not in the expected 2D array format.</p>';
    }

    // Replace {{A1}}, {{B2}}, etc. with corresponding cell values
    $output = preg_replace_callback('/{{([A-Z]+[1-9][0-9]*)}}/', function ($matches) use ($sheet_data) {
        $cell = $matches[1];

        // Convert A1 notation to row and column indices
        $column_letters = preg_replace('/[^A-Z]/', '', $cell);
        $row_number = preg_replace('/[^0-9]/', '', $cell);

        $column_index = 0;
        $length = strlen($column_letters);
        for ($i = 0; $i < $length; $i++) {
            $column_index *= 26;
            $column_index += ord($column_letters[$i]) - ord('A') + 1;
        }
        $column_index--; // Convert to zero-based index
        $row_index = $row_number - 1; // Convert to zero-based index

        if (!isset($sheet_data[$row_index][$column_index])) {
            return '<p>Cell not found in the sheet data.</p>';
        }

        $cell_value = $sheet_data[$row_index][$column_index];
        return esc_html($cell_value);
    }, $content);

    return $output;
}

add_shortcode('sheet2db', 'sheet2db_shortcode');
add_shortcode('sheet2db_table', 'sheet2db_table_shortcode');
add_shortcode('sheet2db_cell', 'sheet2db_cell_shortcode');
?>
